/***************************************************************************
*
* Copyright 2010, 2011 BMW Car IT GmbH
* Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
* Copyright 2013 Codethink Limited
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
* SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
* FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
* SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
* RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
* CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
* CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*
****************************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <linux/input.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <libudev.h>
#include <mtdev.h>
#include <math.h>
#include <errno.h>
#include "InputManager.h"
#include "WindowSystems/InputEventProcessor.h"
#include "WindowSystems/WaylandEvdevInputEvent.h"

#include <algorithm>

#define ARRAY_LENGTH(a) (sizeof (a) / sizeof (a)[0])

#define WL_UNUSED(A) (A)=(A)
static const char default_seat[] = "seat0";
static const char ignored_seat[] = "seat-ign";

// copied from udev/extras/input_id/input_id.c
// we must use this kernel-compatible implementation
#define BITS_PER_LONG (sizeof(unsigned long) * 8)
#define NBITS(x) ((((x)-1)/BITS_PER_LONG)+1)
#define OFF(x)  ((x)%BITS_PER_LONG)
#define BIT(x)  (1UL<<OFF(x))
#define LONG(x) ((x)/BITS_PER_LONG)
#define TEST_BIT(array, bit)    ((array[LONG(bit)] >> OFF(bit)) & 1)
// end copied

#define MAX_VELOCITY_DIFF    1.0
#define MOTION_TIMEOUT       300 // (ms)
#define NUM_POINTER_TRACKERS 16

using namespace std;
using namespace LayerManagerCalibration;
using namespace InputEventProcessing;

struct evdev_dispatch_interface touchpad_interface = {
    WaylandEvdevInputEvent::touchpadProcess,
    WaylandEvdevInputEvent::touchpadDestroy,
};

struct evdev_dispatch_interface fallback_interface = {
    WaylandEvdevInputEvent::fallbackProcess,
    WaylandEvdevInputEvent::fallbackDestroy,
};

// Function prototypes
void acceleratorFilter(struct motion_filter *, struct motion_params *, void *, uint32_t);
void acceleratorDestroy(struct motion_filter *);

struct motion_filter_interface accelerator_interface = {
    acceleratorFilter,
    acceleratorDestroy,
};

static struct touchpad_model_spec touchpad_spec_table[] = {
    {0x0002, 0x0007, TOUCHPAD_MODEL_SYNAPTICS},
    {0x0002, 0x0008, TOUCHPAD_MODEL_ALPS},
    {0x05ac, 0x0000, TOUCHPAD_MODEL_APPLETOUCH},
    {0x0002, 0x000e, TOUCHPAD_MODEL_ELANTECH},
    {0x0000, 0x0000, TOUCHPAD_MODEL_UNKNOWN}
};

////////////////////////////////////////////////////////////////////////////

static int
isMotionEvent(struct input_event *e)
{
    switch (e->type)
    {
    case EV_REL:
        switch (e->code)
        {
        case REL_X:
        case REL_Y:
            return 1;
        }
        break;
    case EV_ABS:
        switch (e->code)
        {
        case ABS_X:
        case ABS_Y:
        case ABS_MT_POSITION_X:
        case ABS_MT_POSITION_Y:
            return 1;
        }
        break;
    }
    return 0;
}

static enum touchpad_model
getTouchpadModel(struct evdev_input_device *device)
{
    struct input_id id;
    unsigned int i;

    if (ioctl(device->fd, EVIOCGID, &id) < 0)
        return TOUCHPAD_MODEL_UNKNOWN;

    for (i = 0; i < sizeof(touchpad_spec_table); ++i)
    {
        if (touchpad_spec_table[i].vendor == id.vendor &&
            (!touchpad_spec_table[i].product ||
            touchpad_spec_table[i].product == id.product))
        {
            return touchpad_spec_table[i].model;
        }
    }

    return TOUCHPAD_MODEL_UNKNOWN;
}

static void
configureTouchpadPressure(struct touchpad_dispatch *touchpad,
                            int32_t pressure_min, int32_t pressure_max)
{
    int32_t range = pressure_max - pressure_min + 1;
    touchpad->has_pressure = 1;

    // Magic numbers from xf86-input-synaptics
    switch (touchpad->model)
    {
    case TOUCHPAD_MODEL_ELANTECH:
        touchpad->pressure.touch_low = pressure_min + 1;
        touchpad->pressure.touch_high = pressure_min + 1;
        break;
    default:
        touchpad->pressure.touch_low = pressure_min + range * (25.0 / 256.0);
        touchpad->pressure.touch_high = pressure_min + range * (30.0 / 256.0);
        break;
    }

    touchpad->pressure.press = pressure_min + range;
}

static double
touchpadProfile(struct motion_filter *filter, void *data,
                double velocity, uint32_t time)
{
    WL_UNUSED(filter);
    WL_UNUSED(time);
    struct touchpad_dispatch *touchpad = (struct touchpad_dispatch*)data;
    double accel_factor;

    accel_factor = velocity * touchpad->constant_accel_factor;

    if (accel_factor > touchpad->max_accel_factor)
        accel_factor = touchpad->max_accel_factor;
    else if (accel_factor < touchpad->min_accel_factor)
        accel_factor = touchpad->min_accel_factor;

    return accel_factor;
}

static motion_filter*
createPointerAccelatorFilter(accel_profile_func_t profile)
{
    struct pointer_accelerator *filter;

    filter = (struct pointer_accelerator*)malloc(sizeof(*filter));
    if (filter == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent","Memory not allocated");
        return NULL;
    }

    filter->base.interface = &accelerator_interface;
    wl_list_init(&filter->base.link);

    filter->profile = profile;
    filter->last_velocity = 0.0;
    filter->last_dx = 0;
    filter->last_dy = 0;

    filter->trackers = (struct pointer_tracker*)
        calloc(NUM_POINTER_TRACKERS, sizeof(*filter->trackers));
    filter->cur_tracker = 0;

    return &filter->base;
}

static evdev_dispatch*
createFallbackDispatch()
{
    struct evdev_dispatch *dispatch =
        (struct evdev_dispatch*)malloc(sizeof(*dispatch));
    if (dispatch == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent","Memory not allocated");
        return NULL;
    }

    dispatch->interface = &fallback_interface;

    return dispatch;
}

////////////////////////////////////////////////////////////////////////

static void
processTouch(struct evdev_input_device *device, struct input_event *e,
                int screen_width, int screen_height)
{
    (void) screen_width;
    (void) screen_height;
    LOG_DEBUG("WaylandEvdevInputEvent",
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "code=" << e->code << ", "
              "value=" << e->value << ", "
              "pending events flag=" << device->pending_events);

    switch (e->code)
    {
    case ABS_MT_SLOT:
        device->mt.slot = e->value;
        break;
    case ABS_MT_TRACKING_ID:
        if (e->value >= 0)
            device->pending_events |= EVDEV_ABSOLUTE_MT_DOWN;
        else
            device->pending_events |= EVDEV_ABSOLUTE_MT_UP;
        break;
    case ABS_MT_POSITION_X:
        device->mt.x[device->mt.slot] = e->value;
        device->pending_events |= EVDEV_ABSOLUTE_MT_MOTION;
        break;
    case ABS_MT_POSITION_Y:
        device->mt.y[device->mt.slot] = e->value;
        device->pending_events |= EVDEV_ABSOLUTE_MT_MOTION;
        break;
    }
}

static void
processAbsoluteMotion(struct evdev_input_device *device, struct input_event *e,
                        int screen_width, int screen_height)
{
    LOG_DEBUG("WaylandEvdevInputEvent",
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "code=" << e->code << ", "
              "value=" << e->value << ", "
              "screen width=" << screen_width << ", "
              "screen height=" << screen_height << ", "
              "pending events flag=" << device->pending_events);

    switch (e->code) {
    case ABS_X:
        device->abs.x = e->value;
        device->pending_events |= EVDEV_ABSOLUTE_MOTION;
        break;
    case ABS_Y:
        device->abs.y = e->value;
        device->pending_events |= EVDEV_ABSOLUTE_MOTION;
        break;
    // The following is the effective code only from a specific model
    /****
    case ABS_X:
        device->abs.y = (e->value - device->abs.min_x) * screen_height /
                        (device->abs.max_x - device->abs.min_x);
        device->pending_events |= EVDEV_ABSOLUTE_MOTION;
        break;
    case ABS_Y:
        device->abs.x = (e->value - device->abs.min_y) * screen_width /
                        (device->abs.max_y - device->abs.min_y);
        device->abs.x = screen_width - device->abs.x;
        device->pending_events |= EVDEV_ABSOLUTE_MOTION;
        break;
     ****/
    }
}

////////////////////////////////////////////////////////////////////////

static int
hysteresis(int in, int center, int margin)
{
    int diff = in - center;
    if (abs(diff) <= margin)
        return center;

    if (diff > margin)
        return center + diff - margin;
    else if (diff < -margin)
        return center + diff + margin;
    return center + diff;
}

static inline struct touchpad_motion*
motionHistoryOffset(struct touchpad_dispatch *touchpad, int offset)
{
    int offsetIndex =
        (touchpad->motion_index - offset + TOUCHPAD_HISTORY_LENGTH) %
        TOUCHPAD_HISTORY_LENGTH;

    return &touchpad->motion_history[offsetIndex];
}

static double
estimateDelta(int x0, int x1, int x2, int x3)
{
    return (x0 + x1 - x2 - x3) / 4;
}

static void
touchpadGetDelta(struct touchpad_dispatch *touchpad, double *dx, double *dy)
{
    *dx = estimateDelta(motionHistoryOffset(touchpad, 0)->x,
                        motionHistoryOffset(touchpad, 1)->x,
                        motionHistoryOffset(touchpad, 2)->x,
                        motionHistoryOffset(touchpad, 3)->x);
    *dy = estimateDelta(motionHistoryOffset(touchpad, 0)->y,
                        motionHistoryOffset(touchpad, 1)->y,
                        motionHistoryOffset(touchpad, 2)->y,
                        motionHistoryOffset(touchpad, 3)->y);
}

static void
filterDispatch(struct motion_filter *filter, struct motion_params *motion,
                void *data, uint32_t time)
{
    filter->interface->filter(filter, motion, data, time);
}

static void
filterMotion(struct touchpad_dispatch *touchpad,
                double *dx, double *dy, uint32_t time)
{
    struct motion_filter *filter;
    struct motion_params  motion;

    motion.dx = *dx;
    motion.dy = *dy;

    wl_list_for_each(filter, &touchpad->motion_filters, link)
    {
        filterDispatch(filter, &motion, touchpad, time);
    }

    *dx = motion.dx;
    *dy = motion.dy;
}

static int
getDirection(int dx, int dy)
{
    int dir = UNDEFINED_DIRECTION;
    int d1;
    int d2;
    double r;

    if (abs(dx) < 2 && abs(dy) < 2)
    {
        if (dx > 0 && dy > 0)
            dir = S | SE | E;
        else if (dx > 0 && dy < 0)
            dir = N | NE | E;
        else if (dx < 0 && dy > 0)
            dir = S | SW | W;
        else if (dx < 0 && dy < 0)
            dir = N | NW | W;
        else if (dx > 0)
            dir = NW | W | SW;
        else if (dx < 0)
            dir = NE | E | SE;
        else if (dy > 0)
            dir = SE | S | SW;
        else if (dy < 0)
            dir = NE | N | NW;
    }
    else
    {
        // Calculate r within the interval  [0 to 8)
        //
        // r = [0 .. 2π] where 0 is North
        // d_f = r / 2π  ([0 .. 1))
        // d_8 = 8 * d_f
        //
        r = atan2(dy, dx);
        r = fmod(r + 2.5*M_PI, 2*M_PI);
        r *= 4*M_1_PI;

        // Mark one or two close enough octants
        d1 = (int)(r + 0.9) % 8;
        d2 = (int)(r + 0.1) % 8;

        dir = (1 << d1) | (1 << d2);
    }

    return dir;
}

static void
feedTrackers(struct pointer_accelerator *accel,
                double dx, double dy, uint32_t time)
{
    int i;
    int current;
    struct pointer_tracker *trackers = accel->trackers;

    for (i = 0; i < NUM_POINTER_TRACKERS; ++i)
    {
        trackers[i].dx += dx;
        trackers[i].dy += dy;
    }

    current = (accel->cur_tracker + 1) % NUM_POINTER_TRACKERS;
    accel->cur_tracker = current;

    trackers[current].dx = 0.0;
    trackers[current].dy = 0.0;
    trackers[current].time = time;
    trackers[current].dir = getDirection(dx, dy);
}

static struct pointer_tracker*
trackerByOffset(struct pointer_accelerator *accel, unsigned int offset)
{
    unsigned int index =
        (accel->cur_tracker + NUM_POINTER_TRACKERS - offset)
        % NUM_POINTER_TRACKERS;
    return &accel->trackers[index];
}

static double
calculateTrackerVelocity(struct pointer_tracker *tracker, uint32_t time)
{
    int dx;
    int dy;
    double distance;

    dx = tracker->dx;
    dy = tracker->dy;
    distance = sqrt(dx*dx + dy*dy);
    return distance / (double)(time - tracker->time);
}

static double
calculateVelocity(struct pointer_accelerator *accel, uint32_t time)
{
    struct pointer_tracker *tracker;
    double velocity;
    double result = 0.0;
    double initial_velocity;
    double velocity_diff;
    unsigned int offset;

    unsigned int dir = trackerByOffset(accel, 0)->dir;

    // Find first velocity
    for (offset = 1; offset < NUM_POINTER_TRACKERS; offset++)
    {
        tracker = trackerByOffset(accel, offset);

        if (time <= tracker->time)
            continue;

        result = initial_velocity =
            calculateTrackerVelocity(tracker, time);
        if (initial_velocity > 0.0)
            break;
    }

    // Find least recent vector within a timelimit, maximum velocity diff
    // and direction threshold.
    for (; offset < NUM_POINTER_TRACKERS; offset++)
    {
        tracker = trackerByOffset(accel, offset);

        // Stop if too far away in time
        if (time - tracker->time > MOTION_TIMEOUT ||
            tracker->time > time)
            break;

        // Stop if direction changed
        dir &= tracker->dir;
        if (dir == 0)
            break;

        velocity = calculateTrackerVelocity(tracker, time);

        // Stop if velocity differs too much from initial
        velocity_diff = fabs(initial_velocity - velocity);
        if (velocity_diff > MAX_VELOCITY_DIFF)
            break;

        result = velocity;
    }

    return result;
}

static double
accelerationProfile(struct pointer_accelerator *accel,
                    void *data, double velocity, uint32_t time)
{
    return accel->profile(&accel->base, data, velocity, time);
}

static double
calculateAcceleration(struct pointer_accelerator *accel,
                        void *data, double velocity, uint32_t time)
{
    double factor;

    factor = accelerationProfile(accel, data, velocity, time);
    factor += accelerationProfile(accel, data, accel->last_velocity, time);
    factor += 4.0 *
                    accelerationProfile(accel, data,
                                    (accel->last_velocity + velocity) / 2,
                                    time);
    factor = factor / 6.0;
    return factor;
}

static double
softenDelta(double lastDelta, double delta)
{
    if (delta < -1.0 || delta > 1.0)
    {
        if (delta > lastDelta)
            return delta - 0.5;
        else if (delta < lastDelta)
            return delta + 0.5;
    }
    return delta;
}

static void
applySoftening(struct pointer_accelerator *accel,
                struct motion_params *motion)
{
    motion->dx = softenDelta(accel->last_dx, motion->dx);
    motion->dy = softenDelta(accel->last_dy, motion->dy);
}

void
acceleratorFilter(struct motion_filter *filter, struct motion_params *motion,
                    void *data, uint32_t time)
{
    struct pointer_accelerator *accel = (struct pointer_accelerator*)filter;
    double velocity;
    double accel_value;

    feedTrackers(accel, motion->dx, motion->dy, time);
    velocity = calculateVelocity(accel, time);
    accel_value = calculateAcceleration(accel, data, velocity, time);

    motion->dx = accel_value * motion->dx;
    motion->dy = accel_value * motion->dy;

    applySoftening(accel, motion);

    accel->last_dx = motion->dx;
    accel->last_dy = motion->dy;

    accel->last_velocity = velocity;
}

void
acceleratorDestroy(struct motion_filter *filter)
{
    struct pointer_accelerator *accel = (struct pointer_accelerator*)filter;

    free(accel->trackers);
    free(accel);
}

/// WaylandEvdevInputEvent /////////////////////////////////////////////////

WaylandEvdevInputEvent::WaylandEvdevInputEvent(WaylandBaseWindowSystem* windowSystem):
WaylandInputEvent(windowSystem)
, m_udev(NULL)
, m_udevMonitor(NULL)
, m_eventLoop(NULL)
, m_screenWidth(0)
, m_screenHeight(0)
, m_pProcessor(NULL)
, m_commandExecutor(windowSystem->getCommandExecutor())
, m_deviceAgainTimer(NULL)
, m_pendingDevices()
, m_windowSystem(windowSystem)
, m_wlIpEventList()
{
    wl_list_init(&m_deviceList);

    m_pProcessor = new InputEventProcessor(m_commandExecutor.getInputDeviceConfigurations());
    m_wlIpEventList.push_back((WaylandInputEvent*)this);

    LOG_INFO("WaylandEvdevInputEvent",
             "Got input device configurations, size="
             << m_commandExecutor.getInputDeviceConfigurations()->size());

    m_eventLoop = wl_event_loop_create();
    if (m_eventLoop == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent", "Failed to create event loop");
    }
    m_deviceAgainTimer = wl_event_loop_add_timer(wl_display_get_event_loop(m_windowSystem->getNativeDisplayHandle()),
                                                 WaylandEvdevInputEvent::onDeviceTimeout, this);
    initUdevMonitor();
}

WaylandEvdevInputEvent::~WaylandEvdevInputEvent()
{
    if (m_udev)
    {
        udev_unref(m_udev);
    }

    struct evdev_input_device *device, *next;

    wl_list_for_each_safe(device, next, &m_deviceList, link)
    {
        removeDevice(device);
    }

    if (m_pProcessor)
    {
        delete m_pProcessor;
    }

    if (m_udevMonitor)
    {
        udev_monitor_unref(m_udevMonitor);
    }

    if (m_eventLoop)
    {
        wl_event_loop_destroy(m_eventLoop);
    }

    PendingInputDeviceList::iterator it = m_pendingDevices.begin();
    while (it != m_pendingDevices.end())
    {
        free((*it)->devnode);
        free(*it);
        it++;
    }
}

int
WaylandEvdevInputEvent::dispatchEvents(int fd, uint32_t mask, void *data)
{
    (void) fd;
    (void) mask;
    WaylandEvdevInputEvent *inputEvent =
        static_cast<WaylandEvdevInputEvent*>(data);

    wl_event_loop_dispatch(inputEvent->m_eventLoop, 0);
    return 1;
}

int
WaylandEvdevInputEvent::handleMonitorEvent(int fd, uint32_t mask, void *data)
{
    (void) fd;
    (void) mask;

    WaylandEvdevInputEvent *inputEvent =
        static_cast<WaylandEvdevInputEvent*>(data);
    struct udev_device *device =
        udev_monitor_receive_device(inputEvent->m_udevMonitor);

    if (!device)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Failed to receive udev device");
        return 1;
    }
    const char *path = udev_device_get_devpath(device);
    const char *node = udev_device_get_devnode(device);
    const char *sysname = udev_device_get_sysname(device);
    const char *action = udev_device_get_action(device);

    const char *device_seat = udev_device_get_property_value(device,
                                                             "ID_SEAT");
    if (!device_seat)
    {
        device_seat = default_seat;
    }

    LOG_DEBUG("WaylandEvdevInputEvent",
              "Device path=" << path << ", "
              "Device node=" << node << ", "
              "System name=" << sysname << ", "
              "Action=" << action << ", Device Seat=" << device_seat);

    if ((strncmp("event", sysname, 5) == 0) &&
        (strcmp(ignored_seat, device_seat) != 0))
    {
        if (strcmp("add", action) == 0)
        {
            bool foundDevice = false;
            struct evdev_input_device *evdevDevice;
            wl_list_for_each(evdevDevice, &inputEvent->m_deviceList, link)
            {
                if (strcmp(node, evdevDevice->devnode) == 0)
                {
                    foundDevice = true;
                }
            }
            if (!foundDevice)
            {
                inputEvent->addDevice(device);
            }
            else
            {
                LOG_WARNING("WaylandEvdevInputEvent",
                            "Can't add device: One already exists for devnode="
                            << node);
            }
        }
        else if (strcmp("remove", action) == 0)
        {
            inputEvent->removeUdevDevice(device);
        }
    }

    udev_device_unref(device);
    return 1;
}

void
WaylandEvdevInputEvent::initUdevMonitor()
{
	struct wl_event_loop *eventLoop;

    if (!m_udev)
        m_udev = udev_new();

    if (!m_udev)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Failed to initialize udev context");
        return;
    }

    m_udevMonitor = udev_monitor_new_from_netlink(m_udev, "udev");
    if (m_udevMonitor == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Failed to initialize udev monitor");
        return;
    }

    int errcode;
    char buf[256];
    errcode = udev_monitor_filter_add_match_subsystem_devtype(m_udevMonitor,
                                                              "input",
                                                              NULL);
    if (errcode != 0)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Failed to add filter to udev monitor, error code="
                    << strerror_r(errcode, buf, 256));
        udev_monitor_unref(m_udevMonitor);
        return;
    }
    errcode = udev_monitor_enable_receiving(m_udevMonitor);
    if (errcode != 0)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Failed to enable receiving on udev monitor, error code="
                    << strerror_r(errcode, buf, 256));
        udev_monitor_unref(m_udevMonitor);
        return;
    }
    eventLoop =
            wl_display_get_event_loop(m_windowSystem->getNativeDisplayHandle());
    if (eventLoop == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "wl display eventLoop is NULL");
        udev_monitor_unref(m_udevMonitor);
        return;
    }

    wl_event_loop_add_fd(eventLoop,
                         udev_monitor_get_fd(m_udevMonitor),
                         WL_EVENT_READABLE,
                         WaylandEvdevInputEvent::handleMonitorEvent,
                         (void*) this);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "Initialized, m_udev=" << m_udev << ", "
              "m_udevMonitor=" << m_udevMonitor);
}

void
WaylandEvdevInputEvent::removeUdevDevice(struct udev_device *udevDevice)
{
    bool foundDevice = false;
    struct evdev_input_device *device;
    wl_list_for_each(device, &m_deviceList, link)
    {
        if (strcmp(device->devnode, udev_device_get_devnode(udevDevice)) == 0)
        {
            foundDevice = true;
            removeDevice(device);
            break;
        }
    }

    if (!foundDevice)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Can't remove device: No device found for devnode="
                    << udev_device_get_devnode(udevDevice));
    }
    else
    {
        LOG_DEBUG("WaylandEvdevInputEvent",
                  "Removed device, "
                  "devnode=" << udev_device_get_devnode(udevDevice));
    }
}

void
WaylandEvdevInputEvent::removeDevice(struct evdev_input_device *device)
{
    struct evdev_dispatch *dispatch = device->dispatch;
    ilmInputDevice ilmDevice = 0;

    LOG_INFO("WaylandEvdevInputEvent",
              "Removed device, name=" << device->deviceName);

    InputManager * pInpManager = \
            device->pWlInputEvent->windowSystem().getInputManager(device->seatName);
    if (NULL != pInpManager)
    {
        if ((device->caps & (EVDEV_MOTION_ABS | EVDEV_MOTION_REL | EVDEV_BUTTON)))
        {
            ilmDevice |= ILM_INPUT_DEVICE_POINTER;
        }
        if ((device->caps & (EVDEV_KEYBOARD)))
        {
            ilmDevice |= ILM_INPUT_DEVICE_KEYBOARD;
        }
        if ((device->caps & (EVDEV_TOUCH)))
        {
            ilmDevice |= ILM_INPUT_DEVICE_TOUCH;
        }
        pInpManager->removeDevices(ilmDevice);
    }

    if (dispatch)
        dispatch->interface->destroy(dispatch);

    wl_event_source_remove(device->source);
    wl_event_source_remove(device->fd_source);

    wl_list_remove(&device->link);
    if (device->mtdev)
        mtdev_close_delete(device->mtdev);
    close(device->fd);
    free(device->devnode);
    free(device->deviceName);
    free(device);
}

void
WaylandEvdevInputEvent::setupInputEvent()
{
    WaylandInputEvent::setupInputEvent();
    m_screenWidth = m_windowSystem->getWindowWidth();
    m_screenHeight = m_windowSystem->getWindowHeight();
    LOG_DEBUG("WaylandEvdevInputEvent",
              "screen width=" << m_screenWidth << ", "
              "height=" << m_screenHeight );
    do {
        bool bRet = addDevices();
        if (!bRet)
        {
            break;
        }
    } while (0);

}

bool
WaylandEvdevInputEvent::addDevices()
{
    if (!m_udev)
        m_udev = udev_new();
    if (!m_udev)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Failed to initialize udev context");
        return false;
    }

    struct udev_enumerate *e = udev_enumerate_new(m_udev);
    udev_enumerate_add_match_subsystem(e, "input");

    udev_enumerate_scan_devices(e);

    struct udev_list_entry *entry;
    const char *path, *sysname, *device_seat;
    struct udev_device *device;
    udev_list_entry_foreach(entry, udev_enumerate_get_list_entry(e))
    {
        path = udev_list_entry_get_name(entry);

        device = udev_device_new_from_syspath(m_udev, path);
        if (!device)
        {
            continue;
        }

        sysname = udev_device_get_sysname(device);

        LOG_DEBUG("WaylandEvdevInputEvent",
                  "Device path=" << path << ", System name=" << sysname);

        device_seat = udev_device_get_property_value(device, "ID_SEAT");
        if (device_seat == NULL)
        {
            device_seat = default_seat;
        }
        if ((strncmp("event", sysname, 5) != 0) ||
            (strcmp(ignored_seat, device_seat) == 0))
        {
            udev_device_unref(device);
            continue;
        }

        addDevice(device);
        udev_device_unref(device);
    }
    udev_enumerate_unref(e);

    if (wl_list_empty(&m_deviceList))
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "No input devices on entering service");
    }

    return true;
}

void
WaylandEvdevInputEvent::addDevice(struct udev_device *udevDevice)
{
    const char *devnode;
    const char *device_seat;
    const char *associated_display;
    int displayID;

    device_seat = udev_device_get_property_value(udevDevice, "ID_SEAT");
    if (!device_seat)
        device_seat = default_seat;
    associated_display = udev_device_get_property_value(udevDevice, "ID_ASSOCIATED_DISPLAY");

    if (!associated_display)
        displayID = INVALID_DISPLAY_ID;
    else
        displayID = atoi(associated_display);

    devnode = udev_device_get_devnode(udevDevice);
    createInputDevice(devnode, device_seat, displayID);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "devnode=" << devnode << ", device_seat=" << device_seat);
}

static bool evdev_device_compare_string(const char* left, evdev_input_device* right)
{
    return strcmp(left, right->devnode) == 0;
}

int WaylandEvdevInputEvent::onDeviceTimeout(void *data)
{
    WaylandEvdevInputEvent *self = static_cast<WaylandEvdevInputEvent*>(data);
    if (not self->m_pendingDevices.empty())
    {
        const char *devname = self->m_pendingDevices.front()->devnode;
        LOG_DEBUG("WaylandEvdevInputEvent",
                  "Rescheduling open for device=" << devname);
        self->createInputDevice(devname, NULL, INVALID_DISPLAY_ID);
    }

    return 1;
}

void
WaylandEvdevInputEvent::createInputDevice(const char *path, const char *seatName, int displayID)
{
    struct evdev_input_device *device;
    struct wl_event_loop *eventLoop;

    PendingInputDeviceList::iterator it =
            std::find_if(m_pendingDevices.begin(),
                         m_pendingDevices.end(),
                         std::bind1st(std::ptr_fun(evdev_device_compare_string), path));
    if (it == m_pendingDevices.end())
    {
        device = static_cast<struct evdev_input_device*>(calloc(sizeof(struct evdev_input_device), 1));
        if (device == NULL)
        {
            LOG_WARNING("WaylandEvdevInputEvent", "Unable to allocate device");
            return;
        }

        device->master = this;
        device->devnode = strdup(path);
        device->mt.slot = -1;
        device->st_slot = MAX_SLOTS;
        device->displayID = displayID;
    }
    else
    {
        device = *it;
        m_pendingDevices.erase(it);
    }
    if (NULL != seatName)
    {
        strncpy(device->seatName, seatName, (MAX_SEATNAME_LNGTH - 2));
        device->seatName[MAX_SEATNAME_LNGTH - 1] = '\0';
    }

    errno = 0;
    device->fd = open(path, O_RDWR | O_NONBLOCK);
    if (device->fd < 0)
    {
        if (errno == EAGAIN && device->retries < MAX_DEVICE_OPEN_RETRIES)
        {
            device->retries++;
            m_pendingDevices.push_back(device);
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "Failed to open device path=" << path <<", "
                      "retrying in 100ms");
            wl_event_source_timer_update(m_deviceAgainTimer, DEVICE_OPEN_RETRY_TIMEOUT_MS);
            return;
        }
        else
        {
            // Failed to open the device for a very long time.
            if (errno == EAGAIN)
            {
                LOG_WARNING("WaylandEvdevInputEvent",
                            "Failed to open device path=" << path <<
                            " for at least " << MAX_DEVICE_OPEN_RETRIES * DEVICE_OPEN_RETRY_TIMEOUT_MS << "ms."
                            << "Please fix the driver.");
            }
            goto err0;
        }
    }

    LOG_DEBUG("WaylandEvdevInputEvent",
              "Device created, device address=" << device << ", "
              "devnode=" << device->devnode << ", "
              "fd path=" << path);

    if (configureDevice(device) < 0)
    {
        goto err1;
    }

    // If the dispatch was not set up use the fallback
    if (device->dispatch == NULL)
        device->dispatch = createFallbackDispatch();
    if (device->dispatch == NULL)
        goto err1;

    if (device->isMt)
    {
        device->mtdev = mtdev_new_open(device->fd);
        if (!device->mtdev)
        {
            LOG_WARNING("WaylandEvdevInputEvent",
                        "mtdev failed to open for device, fd path=" << path);
        }
    }

    if (m_eventLoop == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "m_eventLoop is NULL");
        goto err1;
    }

    eventLoop =
        wl_display_get_event_loop(m_windowSystem->getNativeDisplayHandle());
    if (eventLoop == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent", "Display has no event loop");
        goto err1;
    }

    device->source = wl_event_loop_add_fd(m_eventLoop, device->fd, WL_EVENT_READABLE,
                                            WaylandEvdevInputEvent::handleInputEvent,
                                            device);

    device->fd_source = wl_event_loop_add_fd(eventLoop, device->fd, WL_EVENT_READABLE,
                         WaylandEvdevInputEvent::dispatchEvents,
                         this);

    if (device->source == NULL || device->fd_source == NULL)
    {
        goto err2;
    }

    wl_list_insert(m_deviceList.prev, &(device->link));
    return;

err2:
    LOG_WARNING("WaylandEvdevInputEvent",
                "Unable to set device source, fd path=" << path);
    device->dispatch->interface->destroy(device->dispatch);
err1:
    LOG_WARNING("WaylandEvdevInputEvent",
                "Unable to create device dispatcher, fd path=" << path);
    close(device->fd);
err0:
    LOG_WARNING("WaylandEvdevInputEvent",
                "Unable to open device, fd path=" << path);
    free(device->devnode);
    free(device);
    return;
}

int
WaylandEvdevInputEvent::configureDevice(struct evdev_input_device *device)
{
    struct input_absinfo absinfo;
    unsigned long ev_bits[NBITS(EV_MAX)];
    unsigned long abs_bits[NBITS(ABS_MAX)];
    unsigned long rel_bits[NBITS(ABS_MAX)];
    unsigned long key_bits[NBITS(KEY_MAX)];
    InputManager* pInpManager = NULL;

    int hasKey = 0;
    int hasAbs = 0;
    int i;
    device->caps = 0;

    char device_name[256];

    // Read device name from evdev
    if (ioctl(device->fd, EVIOCGNAME(sizeof(device_name)), device_name) < 0)
    {
        device->deviceName = strdup("");
    }
    else
    {
        device->deviceName = strdup(device_name);
    }

    ioctl(device->fd, EVIOCGBIT(0, sizeof(ev_bits)), ev_bits);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "Device configured, "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "devnode=" << device->devnode << ", "
              "ev_bits=" << ev_bits);
    if (TEST_BIT(ev_bits, EV_ABS))
    {
        hasAbs = 1;

        ioctl(device->fd, EVIOCGBIT(EV_ABS, sizeof(abs_bits)), abs_bits);

        if (TEST_BIT(abs_bits, ABS_MT_POSITION_X))
        {
            ioctl(device->fd, EVIOCGABS(ABS_MT_POSITION_X), &absinfo);
            device->abs.min_x = absinfo.minimum;
            device->abs.max_x = absinfo.maximum;
            device->caps |= EVDEV_MOTION_ABS;
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "Device configured, "
                      "device name=" << device->deviceName << ", "
                      "device address=" << device << ", "
                      "devnode=" << device->devnode << ", "
                      "abs min x=" << device->abs.min_x << ", "
                      "abs max x=" << device->abs.max_x);
        }


        if (TEST_BIT(abs_bits, ABS_MT_POSITION_Y))
        {
            ioctl(device->fd, EVIOCGABS(ABS_MT_POSITION_Y), &absinfo);
            device->abs.min_y = absinfo.minimum;
            device->abs.max_y = absinfo.maximum;
            device->caps |= EVDEV_MOTION_ABS;
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "Device configured, "
                      "device name=" << device->deviceName << ", "
                      "device address=" << device << ", "
                      "devnode=" << device->devnode << ", "
                      "abs min y=" << device->abs.min_y << ", "
                      "abs max y=" << device->abs.max_y);
        }

        if (TEST_BIT(abs_bits, ABS_X))
        {
            ioctl(device->fd, EVIOCGABS(ABS_X), &absinfo);
            device->abs.min_x = absinfo.minimum;
            device->abs.max_x = absinfo.maximum;
            device->caps |= EVDEV_MOTION_ABS;
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "Device configured, "
                      "device name=" << device->deviceName << ", "
                      "device address=" << device << ", "
                      "devnode=" << device->devnode << ", "
                      "abs min x=" << device->abs.min_x << ", "
                      "abs max x=" << device->abs.max_x);
        }
        if (TEST_BIT(abs_bits, ABS_Y))
        {
            ioctl(device->fd, EVIOCGABS(ABS_Y), &absinfo);
            device->abs.min_y = absinfo.minimum;
            device->abs.max_y = absinfo.maximum;
            device->caps |= EVDEV_MOTION_ABS;
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "Device configured, "
                      "device name=" << device->deviceName << ", "
                      "device address=" << device << ", "
                      "devnode=" << device->devnode << ", "
                      "abs min y=" << device->abs.min_y << ", "
                      "abs max y=" << device->abs.max_y);
        }

        if (TEST_BIT(abs_bits, ABS_MT_SLOT))
        {
            device->isMt = 1;
            device->mt.slot = 0;
            device->caps |= EVDEV_TOUCH;
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "Multi-touch set, "
                      "device name=" << device->deviceName << ", "
                      "device address=" << device << ", "
                      "devnode=" << device->devnode);
        }

        if ((device->abs.max_x == 0 && device->abs.min_x == 0) ||
            (device->abs.max_y == 0 && device->abs.min_y == 0))
        {
            LOG_ERROR("WaylandEvdevInputEvent",
                      "Device configuration issue, got no valid max/min values");
            return -1;
        }
    }
    if (TEST_BIT(ev_bits, EV_REL))
    {
        ioctl(device->fd, EVIOCGBIT(EV_REL, sizeof(rel_bits)), rel_bits);
        if (TEST_BIT(rel_bits, REL_X) || TEST_BIT(rel_bits, REL_Y)
            || TEST_BIT(rel_bits, REL_WHEEL) || TEST_BIT(rel_bits, REL_HWHEEL))
        {
            device->caps |= EVDEV_MOTION_REL;
        }
    }
    if (TEST_BIT(ev_bits, EV_KEY))
    {
        hasKey = 1;
        ioctl(device->fd, EVIOCGBIT(EV_KEY, sizeof(key_bits)), key_bits);
        if (TEST_BIT(key_bits, BTN_TOOL_FINGER) &&
            !TEST_BIT(key_bits, BTN_TOOL_PEN) && hasAbs)
        {
            device->dispatch = createTouchpad(device);
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "Touchpad dispatch created, "
                      "device name=" << device->deviceName << ", "
                      "device address=" << device << ", "
                      "devnode=" << device->devnode << ", "
                      "dispatch=" << device->dispatch);
        }

        for (i = KEY_ESC; i < KEY_MAX; ++i)
        {
            if (i >= BTN_MISC && i < KEY_OK)
                continue;
            if (TEST_BIT(key_bits, i))
            {
                device->caps |= EVDEV_KEYBOARD;
                break;
            }
        }
        for (i = BTN_MISC; i < KEY_OK; ++i)
        {
            if (TEST_BIT(key_bits, i))
            {
                device->caps |= EVDEV_BUTTON;
                break;
            }
        }
    }
    if (TEST_BIT(ev_bits, EV_LED))
    {
        device->caps |= EVDEV_KEYBOARD;
    }

    // This rule tries to catch accelerometer devices and opt out. We may
    // want to adjust the protocol later adding a proper event for dealing
    // with accelerometers and implement here accordingly
    if (hasAbs && !hasKey && !device->isMt)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Opting out accelerometer device, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "devnode=" << device->devnode);
        return -1;
    }

#if 0
    fprintf(stdout, "DEVICE: [%s] information\n", device->devnode);
    fprintf(stdout, "        capabilities: EVDEV_KEYBOARD   %s\n"
                    "                      EVDEV_BUTTON     %s\n"
                    "                      EVDEV_MOTION_ABS %s\n"
                    "                      EVDEV_MOTION_REL %s\n"
                    "                      EVDEV_TOUCH      %s\n",
            (device->caps & EVDEV_KEYBOARD)   ? "TRUE" : "FALSE",
            (device->caps & EVDEV_BUTTON)     ? "TRUE" : "FALSE",
            (device->caps & EVDEV_MOTION_ABS) ? "TRUE" : "FALSE",
            (device->caps & EVDEV_MOTION_REL) ? "TRUE" : "FALSE",
            (device->caps & EVDEV_TOUCH)      ? "TRUE" : "FALSE");
    if (device->caps & EVDEV_MOTION_ABS)
    {
        fprintf(stdout, "        abs: min_x(%4d), min_y(%4d)\n", device->abs.min_x, device->abs.min_y);
        fprintf(stdout, "             max_x(%4d), max_y(%4d)\n", device->abs.max_x, device->abs.max_y);
        fprintf(stdout, "                 x(%4d),     y(%4d)\n", device->abs.x, device->abs.y);
    }
    fprintf(stdout, "\n");
#endif

    // enable pointer, keyboard and touch support for input devices
    // with a configuration because we need to be able to generate
    // pointer, key and touch events
    if (m_pProcessor->deviceNeedsProcessing(device->deviceName))
    {
        device->caps |= EVDEV_MOTION_ABS | EVDEV_BUTTON | EVDEV_KEYBOARD | EVDEV_TOUCH;

        LOG_DEBUG("WaylandEvdevInputEvent",
                  "Device handled by input event processor, "
                  "device name=" << device->deviceName << ", "
                  "device address=" << device << ", "
                  "devnode=" << device->devnode);
    }

    WlInputEventList::iterator it;
    bool seatexists = false;
    WaylandInputEvent *pWlIpEvent;
    for (it = m_wlIpEventList.begin(); 
        (it != m_wlIpEventList.end()) && (false == seatexists); it++)
    {
        if (0 == strcmp((*it)->getSeatName(), device->seatName))
        {
            pWlIpEvent = *it;
            seatexists = true;
        }
    }
    pInpManager = m_windowSystem->getInputManager(device->seatName);
    if (false == seatexists)
    {
        if (NULL == pInpManager)
        {
            pInpManager = m_windowSystem->addInputManager(device->seatName);
        }
        pWlIpEvent = new WaylandInputEvent(m_windowSystem, device->seatName, pInpManager);
        if (NULL == pWlIpEvent)
        {
            LOG_ERROR("WaylandEvdevInputEvent",
                      "Memory allocation failed for wayland input event");
            return -1;
        }
        pWlIpEvent->setupInputEvent();
        m_wlIpEventList.push_back(pWlIpEvent);
        pWlIpEvent->setSeatName(device->seatName);
    }
    if (NULL == pInpManager)
    {
        /*Obtain default input manager*/
        pInpManager = m_windowSystem->getInpManager();
    }
    if ((device->caps & (EVDEV_MOTION_ABS | EVDEV_MOTION_REL | EVDEV_BUTTON)))
    {
        pWlIpEvent->initPointerDevice();
        pInpManager->addDevices(ILM_INPUT_DEVICE_POINTER);
    }
    if ((device->caps & (EVDEV_KEYBOARD)))
    {
        pWlIpEvent->initKeyboardDevice(NULL);
        pInpManager->addDevices(ILM_INPUT_DEVICE_KEYBOARD);
    }
    if ((device->caps & (EVDEV_TOUCH)))
    {
        pWlIpEvent->initTouchDevice();
        pInpManager->addDevices(ILM_INPUT_DEVICE_TOUCH);
    }
    device->pWlInputEvent = pWlIpEvent;
    LOG_INFO("WaylandEvdevInputEvent",
              "Device configured, "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "devnode=" << device->devnode << ", "
              "capabilities set to=" << device->caps);

    return 0;
}

struct evdev_dispatch*
WaylandEvdevInputEvent::createTouchpad(struct evdev_input_device *device)
{
    struct touchpad_dispatch *touchpad;

    touchpad = (struct touchpad_dispatch*)malloc(sizeof(*touchpad));
    if (touchpad == NULL)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Unable to create touchpad for device, "
                    "name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "devnode=" << device->devnode);
        return NULL;
    }
    else
    {
        LOG_DEBUG("WaylandEvdevInputEvent",
                  "Created touchpad for device, "
                  "name=" << device->deviceName << ", "
                  "device address=" << device << ", "
                  "devnode=" << device->devnode);
    }

    touchpad->base.interface = &touchpad_interface;

    touchpad->device = device;
    wl_list_init(&touchpad->motion_filters);

    configureTouchpad(touchpad, device);

    return &touchpad->base;
}

void
WaylandEvdevInputEvent::configureTouchpad(struct touchpad_dispatch *touchpad,
                                            struct evdev_input_device *device)
{
    struct motion_filter *accel;

    struct input_absinfo absinfo;
    unsigned long abs_bits[NBITS(ABS_MAX)];

    double width;
    double height;
    double diagonal;

    // Detect model
    touchpad->model = getTouchpadModel(device);

    // Configure pressure
    ioctl(device->fd, EVIOCGBIT(EV_ABS, sizeof(abs_bits)), abs_bits);
    if (TEST_BIT(abs_bits, ABS_PRESSURE))
    {
        ioctl(device->fd, EVIOCGABS(ABS_PRESSURE), &absinfo);
        configureTouchpadPressure(touchpad,
                                absinfo.minimum,
                                absinfo.maximum);
    }

    // Configure acceleration factor
    width = abs(device->abs.max_x - device->abs.min_x);
    height = abs(device->abs.max_y - device->abs.min_y);
    diagonal = sqrt(width*width + height*height);

    touchpad->constant_accel_factor =
        DEFAULT_CONSTANT_ACCEL_NUMERATOR / diagonal;

    touchpad->min_accel_factor = DEFAULT_MIN_ACCEL_FACTOR;
    touchpad->max_accel_factor = DEFAULT_MAX_ACCEL_FACTOR;

    touchpad->hysteresis.margin_x = diagonal / DEFAULT_HYSTERESIS_MARGIN_DENOMINATOR;
    touchpad->hysteresis.margin_y = diagonal / DEFAULT_HYSTERESIS_MARGIN_DENOMINATOR;
    touchpad->hysteresis.center_x = 0;
    touchpad->hysteresis.center_y = 0;

    // Configure acceleration profile
    accel = createPointerAccelatorFilter(touchpadProfile);
    wl_list_insert(&touchpad->motion_filters, &accel->link);

    // Setup initial state
    touchpad->reset = 1;

    memset(touchpad->motion_history, 0, sizeof touchpad->motion_history);
    touchpad->motion_index = 0;
    touchpad->motion_count = 0;

    touchpad->state = TOUCHPAD_STATE_NONE;
    touchpad->last_finger_state = 0;
    touchpad->finger_state = 0;

    LOG_DEBUG("WaylandEvdevInputEvent",
              "Configured touchpad for device, "
              "name=" << device->deviceName << ", "
              "devnode=" << device->devnode << ", "
              "device address=" << device << ", "
              "constant acceleration factor="
              << touchpad->constant_accel_factor << ", "
              "hysteresis margin x=" << touchpad->hysteresis.margin_x << ", "
              "hysteresis margin y=" << touchpad->hysteresis.margin_y << ", "
              "acceleration profile address=" << &accel->link);
}

int
WaylandEvdevInputEvent::handleInputEvent(int fd, uint32_t mask, void *data)
{
    WL_UNUSED(mask);
    errno = 0;
    int errorNo = 0;
    struct evdev_input_device *device = (evdev_input_device*)data;
    struct input_event ev[32];
    int len;
    do {
        if (device->mtdev)
        {
            len = mtdev_get(device->mtdev, fd, ev,
                ARRAY_LENGTH(ev)) * sizeof(struct input_event);
            errorNo=errno;
        }
        else
        {
            len = read(fd, &ev, sizeof(ev));
            errorNo=errno;
        }

        if (len < 0 || len % sizeof(ev[0]) != 0)
        {
            LOG_DEBUG("WaylandEvdevInputEvent",
                      "No events to process, "
                      "device name=" << device->deviceName << ", "
                      "device address=" << device << ", "
                      "length reported=" << len);

            if ( (errorNo == EBADF) ||
                 (errorNo == ENXIO) ||
                 (errorNo == ENODEV))
            {
                char errbuf[30];
                char *errmsg;
                errmsg = strerror_r(errorNo, errbuf, 30);
                LOG_WARNING("WaylandEvdevInputEvent",
                            "Removing device, address=data, "
                            "error=" << errmsg);
                WaylandEvdevInputEvent::removeDevice(device);
            }
            return 1;
        }

        if(!device->caps)
        {
            LOG_ERROR("WaylandEvdevInputEvent",
                        "Device does not have any capabilities. "
                        "Not processing any events on the device: " << device->deviceName);
            return 1;
        }

        WaylandEvdevInputEvent::processEvents(device, ev, len / sizeof(ev[0]));
    } while (len > 0);

    return 1;
}

void
WaylandEvdevInputEvent::processEvents(struct evdev_input_device *device,
                                        struct input_event *ev,
                                        int count)
{
    struct evdev_dispatch *dispatch = device->dispatch;
    struct input_event *e, *end;
    uint32_t time = 0;

    device->pending_events = 0;

    e = ev;
    end = e + count;

    LOG_DEBUG("WaylandEvdevInputEvent",
              "events start time="
              << (e->time.tv_sec * 1000 + e->time.tv_usec / 1000) << ", "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "number to process=" << count);

    for (; e < end; ++e)
    {
        time = e->time.tv_sec * 1000 + e->time.tv_usec / 1000;

        if (!isMotionEvent(e))
        {
            WaylandEvdevInputEvent::flushMotion(device, time);
        }

        dispatch->interface->process(dispatch, device, e, time);
    }

    LOG_DEBUG("WaylandEvdevInputEvent",
              "events end time=" << time << ", "
              "device name=" << device->deviceName << ", "
              "device address=" << device);

    WaylandEvdevInputEvent::flushMotion(device, time);
}

void
WaylandEvdevInputEvent::flushMotion(struct evdev_input_device *device,
                                    uint32_t time)
{
    if (!device->pending_events)
    {
        LOG_DEBUG("WaylandEvdevInputEvent",
                  "No pending events, " <<
                  "time=" << time << ", "
                  "device name=" << device->deviceName << ", "
                  "device address=" << device);
        return;
    }

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Input event could not be created, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device);
        return;
    }

    bool needs_processing =
        inputEvent->m_pProcessor->deviceNeedsProcessing(device->deviceName);

    if (device->pending_events & EVDEV_RELATIVE_MOTION)
    {
        struct wl_seat *wlSeat = device->pWlInputEvent->inputDevice().seat();
        if (wlSeat)
        {
            notifyMotion(device, time,
                         wlSeat->pointer->x + device->rel.dx,
                         wlSeat->pointer->y + device->rel.dy);
        }
        else
        {
            LOG_WARNING("WaylandEvdevInputEvent",
                        "Seat not set for event, "
                        "device name=" << device->deviceName << ", "
                        "device address=" << device);
        }
        device->pending_events &= ~EVDEV_RELATIVE_MOTION;
        device->rel.dx = 0;
        device->rel.dy = 0;
    }

    if (needs_processing)
    {
        if (device->pending_events & (EVDEV_ABSOLUTE_MT_DOWN
                                      | EVDEV_ABSOLUTE_MT_MOTION
                                      | EVDEV_ABSOLUTE_MT_UP))
        {
            inputEvent->m_pProcessor->processMultiTouchEvent(device, time);
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_DOWN;
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_MOTION;
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_UP;
        }
        if (device->pending_events & EVDEV_ABSOLUTE_MOTION)
        {
            inputEvent->m_pProcessor->processMotionEvent(device, time);
            device->pending_events &= ~EVDEV_ABSOLUTE_MOTION;
        }
        if (device->pending_events != 0)
        {
            LOG_WARNING("WaylandEvdevInputEvent",
                        "Event not valid for processing"
                        "device name=" << device->deviceName << ", "
                        "device address=" << device << ", "
                        "pending events=" << device->pending_events);
            device->pending_events = 0;
        }
    }
    else
    {
        if (device->pending_events & EVDEV_ABSOLUTE_MT_DOWN)
        {
            notifyTouch(device);
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_DOWN;
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_MOTION;
        }
        if (device->pending_events & EVDEV_ABSOLUTE_MT_MOTION)
        {
            notifyTouch(device);
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_DOWN;
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_MOTION;
        }
        if (device->pending_events & EVDEV_ABSOLUTE_MT_UP)
        {
            notifyTouch(device);
            device->pending_events &= ~EVDEV_ABSOLUTE_MT_UP;
        }
    }

    if (device->pending_events & EVDEV_ABSOLUTE_MOTION)
    {
        notifyMotion(device, time,
                     wl_fixed_from_int(device->abs.x),
                     wl_fixed_from_int(device->abs.y));

        device->pending_events &= ~EVDEV_ABSOLUTE_MOTION;
    }
}

/// Default event handler //////////////////////////////////////////////////

void
WaylandEvdevInputEvent::fallbackProcess(struct evdev_dispatch *dispatch,
                                        struct evdev_input_device *device,
                                        struct input_event *e,
                                        uint32_t time)
{
    WL_UNUSED(dispatch);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "dispatch=" << dispatch << ", "
              "event time=" << time << ", "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "event type=" << e->type);

    switch (e->type)
    {
    case EV_REL:
        evdevProcessRelative(device, time, e);
        break;
    case EV_ABS:
        evdevProcessAbsolute(device, e);
        break;
    case EV_KEY:
        evdevProcessKey(device, time, e);
        break;
    }
}

void
WaylandEvdevInputEvent::fallbackDestroy(struct evdev_dispatch *dispatch)
{
    LOG_DEBUG("WaylandEvdevInputEvent",
              "dispatch=" << dispatch);
    if (dispatch) free(dispatch);
}

void
WaylandEvdevInputEvent::evdevProcessRelative(struct evdev_input_device *device,
                                             uint32_t time, struct input_event *e)
{
    LOG_DEBUG("WaylandEvdevInputEvent",
              "event time=" << time << ", "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "event code=" << e->code);

    switch (e->code)
    {
    case REL_X:
        device->rel.dx += wl_fixed_from_int(e->value);
        device->pending_events |= EVDEV_RELATIVE_MOTION;
        break;
    case REL_Y:
        device->rel.dy += wl_fixed_from_int(e->value);
        device->pending_events |= EVDEV_RELATIVE_MOTION;
        break;
    case REL_WHEEL:
        notifyAxis(device, time, WL_POINTER_AXIS_VERTICAL_SCROLL, wl_fixed_from_int(e->value));
        break;
    case REL_HWHEEL:
        notifyAxis(device, time, WL_POINTER_AXIS_HORIZONTAL_SCROLL, wl_fixed_from_int(e->value));
        break;
    default:
        notifyAxis(device, time, e->code, wl_fixed_from_int(e->value));
        break;
    }
}

void
WaylandEvdevInputEvent::evdevProcessAbsolute(struct evdev_input_device *device,
                                             struct input_event *e)
{
    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    int w = 0;
    int h = 0;

    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device);
        return;
    }

    inputEvent->getScreenResolution(device->displayID, w, h);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "associated display=" << device->displayID << ", "
              "screen width=" << w << ", "
              "screen heigth=" << h << ", "
              "multi-touch=" << device->isMt);

    if (device->isMt)
    {
        if ((e->code == ABS_MT_SLOT)
             || (e->code == ABS_MT_TRACKING_ID)
             || (e->code == ABS_MT_POSITION_X)
             || (e->code == ABS_MT_POSITION_Y))
        {
            processTouch(device, e, w, h);
        }
    }
    else if ((e->code == ABS_X) || (e->code == ABS_Y))
    {
        processAbsoluteMotion(device, e, w, h);
    }
}

void
WaylandEvdevInputEvent::evdevProcessKey(struct evdev_input_device *device,
                                        uint32_t time, struct input_event *e)
{
    if (e->value == 2)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Key repeat event ignored, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "event code=" << e->code << ", "
                    "time=" << time << ", "
                    );
        return;
    }

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "event code=" << e->code << ", "
                    "time=" << time);
        return;
    }
    else
    {
        LOG_DEBUG("WaylandEvdevInputEvent",
                  "device name=" << device->deviceName << ", "
                  "device address=" << device << ", "
                  "event code=" << e->code << ", "
                  "event value=" << e->value << ", "
                  "time=" << time);
    }

    /*don't send a button touch from a multitouch device*/
    if ((device->isMt) && (e->code == BTN_TOUCH))
        return;

    switch (e->code)
    {
    case BTN_LEFT:
    case BTN_RIGHT:
    case BTN_MIDDLE:
    case BTN_SIDE:
    case BTN_EXTRA:
    case BTN_FORWARD:
    case BTN_BACK:
    case BTN_TASK:
    case BTN_TOUCH:
        // notify_button
        if (inputEvent->m_pProcessor->deviceNeedsProcessing(device->deviceName))
        {
            inputEvent->m_pProcessor->processButtonEvent(device,
                time,
                e->code,
                e->value ? WL_POINTER_BUTTON_STATE_PRESSED
                         : WL_POINTER_BUTTON_STATE_RELEASED);

        }
        else
        {
            notifyButton(device, time, e->code,
                         e->value ? WL_POINTER_BUTTON_STATE_PRESSED
                                  : WL_POINTER_BUTTON_STATE_RELEASED);
        }
        break;
    default:
        // notify_key
        notifyKey(device, time, e->code,
                e->value ? WL_KEYBOARD_KEY_STATE_PRESSED
                        : WL_KEYBOARD_KEY_STATE_RELEASED,
                  true, true);
    }
}

/// Multi-touch event handler //////////////////////////////////////////////

void
WaylandEvdevInputEvent::touchpadProcess(struct evdev_dispatch *dispatch,
                                        struct evdev_input_device *device,
                                        struct input_event *e,
                                        uint32_t time)
{
    struct touchpad_dispatch *touchpad = (struct touchpad_dispatch*)dispatch;

    LOG_DEBUG("WaylandEvdevInputEvent",
              "dispatch=" << touchpad << ", "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "event code=" << e->code << ", "
              "event value=" << e->value << ", "
              "time=" << time);

    switch (e->type)
    {
    case EV_SYN:
        if (e->code == SYN_REPORT)
            touchpad->event_mask |= TOUCHPAD_EVENT_REPORT;
        break;
    case EV_ABS:
        touchpadProcessAbsolute(touchpad, device, e);
        break;
    case EV_KEY:
        touchpadProcessKey(touchpad, device, e, time);
        break;
    }

    touchpadUpdateState(touchpad, time);
}

void
WaylandEvdevInputEvent::touchpadDestroy(struct evdev_dispatch *dispatch)
{
    struct touchpad_dispatch *touchpad = (struct touchpad_dispatch*)dispatch;
    struct motion_filter *filter;
    struct motion_filter *next;

    LOG_DEBUG("WaylandEvdevInputEvent",
              "dispatch=" << touchpad);

    wl_list_for_each_safe(filter, next, &touchpad->motion_filters, link)
    {
        filter->interface->destroy(filter);
    }

    if (dispatch) free(dispatch);
}

void
WaylandEvdevInputEvent::touchpadProcessAbsolute(struct touchpad_dispatch *touchpad,
                                                struct evdev_input_device *device,
                                                struct input_event *e)
{
    LOG_DEBUG("WaylandEvdevInputEvent",
              "dispatch=" << touchpad << ", "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "event code=" << e->code << ", "
              "event value=" << e->value);

    WL_UNUSED(device);

    switch (e->code)
    {
    case ABS_PRESSURE:
        if (e->value > touchpad->pressure.press)
            touchpad->state = TOUCHPAD_STATE_PRESS;
        else if (e->value > touchpad->pressure.touch_high)
            touchpad->state = TOUCHPAD_STATE_TOUCH;
        else if (e->value < touchpad->pressure.touch_low)
        {
            if (touchpad->state > TOUCHPAD_STATE_NONE)
                touchpad->reset = 1;
            touchpad->state = TOUCHPAD_STATE_NONE;
        }
        break;
    case ABS_X:
        if (touchpad->state >= TOUCHPAD_STATE_TOUCH)
        {
            touchpad->hw_abs.x = e->value;
            touchpad->event_mask |= TOUCHPAD_EVENT_ABSOLUTE_ANY;
            touchpad->event_mask |= TOUCHPAD_EVENT_ABSOLUTE_X;
        }
        break;
    case ABS_Y:
        if (touchpad->state >= TOUCHPAD_STATE_TOUCH)
        {
            touchpad->hw_abs.y = e->value;
            touchpad->event_mask |= TOUCHPAD_EVENT_ABSOLUTE_ANY;
            touchpad->event_mask |= TOUCHPAD_EVENT_ABSOLUTE_Y;
        }
        break;
    }
}

void
WaylandEvdevInputEvent::touchpadProcessKey(struct touchpad_dispatch *touchpad,
                                           struct evdev_input_device * device,
                                           struct input_event *e,
                                           uint32_t time)
{

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "event code=" << e->code << ", "
                    "time=" << time);
        return;
    }

    bool needs_processing = inputEvent->m_pProcessor->deviceNeedsProcessing(device->deviceName);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "dispatch=" << touchpad << ", "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "event code=" << e->code << ", "
              "event value=" << e->value << ", "
              "time=" << time << ", "
              "processing state=" << needs_processing);

    switch (e->code)
    {
    case BTN_TOUCH:
        if (!touchpad->has_pressure)
        {
            if (!e->value)
            {
                touchpad->state = TOUCHPAD_STATE_NONE;
                touchpad->reset = 1;
            }
            else
            {
                touchpad->state = e->value ? TOUCHPAD_STATE_TOUCH
                                           : TOUCHPAD_STATE_NONE;
            }
        }
        break;
    case BTN_LEFT:
    case BTN_RIGHT:
    case BTN_MIDDLE:
    case BTN_SIDE:
    case BTN_EXTRA:
    case BTN_FORWARD:
    case BTN_BACK:
    case BTN_TASK:
        // notify_button
        if (needs_processing)
        {
            inputEvent->m_pProcessor->processButtonEvent(device,
                time,
                e->code,
                e->value ? WL_POINTER_BUTTON_STATE_PRESSED
                         : WL_POINTER_BUTTON_STATE_RELEASED);
        }
        else
        {
            notifyButton(device, time, e->code,
                         e->value ? WL_POINTER_BUTTON_STATE_PRESSED
                                  : WL_POINTER_BUTTON_STATE_RELEASED);
        }
        break;
    case BTN_TOOL_PEN:
    case BTN_TOOL_RUBBER:
    case BTN_TOOL_BRUSH:
    case BTN_TOOL_PENCIL:
    case BTN_TOOL_AIRBRUSH:
    case BTN_TOOL_MOUSE:
    case BTN_TOOL_LENS:
        touchpad->reset = 1;
        break;
    case BTN_TOOL_FINGER:
        touchpad->finger_state =
            ~TOUCHPAD_FINGERS_ONE | e->value ? TOUCHPAD_FINGERS_ONE : 0;
        break;
    case BTN_TOOL_DOUBLETAP:
        touchpad->finger_state =
            ~TOUCHPAD_FINGERS_TWO | e->value ? TOUCHPAD_FINGERS_TWO : 0;
        break;
    case BTN_TOOL_TRIPLETAP:
        touchpad->finger_state =
            ~TOUCHPAD_FINGERS_THREE | e->value ? TOUCHPAD_FINGERS_THREE : 0;
        break;
    }
}

void
WaylandEvdevInputEvent::touchpadUpdateState(struct touchpad_dispatch *touchpad,
                                            uint32_t time)
{
    int motion_index;
    int center_x;
    int center_y;
    double dx;
    double dy;

    LOG_DEBUG("WaylandEvdevInputEvent",
              "dispatch=" << touchpad << ", "
              "time=" << time);

    if (touchpad->reset ||
        touchpad->last_finger_state != touchpad->finger_state)
    {
        touchpad->reset = 0;
        touchpad->motion_count = 0;
        touchpad->event_mask = TOUCHPAD_EVENT_NONE;
        touchpad->event_mask_filter =
            TOUCHPAD_EVENT_ABSOLUTE_X | TOUCHPAD_EVENT_ABSOLUTE_Y;

        touchpad->last_finger_state = touchpad->finger_state;

        return;
    }
    touchpad->last_finger_state = touchpad->finger_state;

    if (!(touchpad->event_mask & TOUCHPAD_EVENT_REPORT))
        return;
    else
        touchpad->event_mask &= ~TOUCHPAD_EVENT_REPORT;

    if ((touchpad->event_mask & touchpad->event_mask_filter) !=
        touchpad->event_mask_filter)
        return;

    touchpad->event_mask_filter = TOUCHPAD_EVENT_ABSOLUTE_ANY;
    touchpad->event_mask = 0;

    // Avoid noice by moving center only when delta reaches a threshold
    // distance from the old center
    if (touchpad->motion_count > 0)
    {
        center_x = hysteresis(touchpad->hw_abs.x,
                      touchpad->hysteresis.center_x,
                      touchpad->hysteresis.margin_x);
        center_y = hysteresis(touchpad->hw_abs.y,
                      touchpad->hysteresis.center_y,
                      touchpad->hysteresis.margin_y);
    }
    else
    {
        center_x = touchpad->hw_abs.x;
        center_y = touchpad->hw_abs.y;
    }
    touchpad->hysteresis.center_x = center_x;
    touchpad->hysteresis.center_y = center_y;
    touchpad->hw_abs.x = center_x;
    touchpad->hw_abs.y = center_y;

    // Update motion history tracker
    motion_index = (touchpad->motion_index + 1) % TOUCHPAD_HISTORY_LENGTH;
    touchpad->motion_index = motion_index;
    touchpad->motion_history[motion_index].x = touchpad->hw_abs.x;
    touchpad->motion_history[motion_index].y = touchpad->hw_abs.y;
    if (touchpad->motion_count < 4)
        touchpad->motion_count++;

    if (touchpad->motion_count >= 4)
    {
        touchpadGetDelta(touchpad, &dx, &dy);

        filterMotion(touchpad, &dx, &dy, time);

        touchpad->device->rel.dx = wl_fixed_from_double(dx);
        touchpad->device->rel.dy = wl_fixed_from_double(dy);
        touchpad->device->pending_events |= EVDEV_RELATIVE_MOTION;
    }

}

/// Notifier ///////////////////////////////////////////////////////////////

void
WaylandEvdevInputEvent::notifyButton(struct evdev_input_device *device,
                                     uint32_t time, int32_t button,
                                     enum wl_pointer_button_state state)
{
    WLEvent         wlEvent;
    struct wl_seat *wlSeat = NULL;
    uint32_t        serial;

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "button=" << button << ", "
                    "state=" << state << ", "
                    "time=" << time);
        return;
    }

    wlSeat = device->pWlInputEvent->inputDevice().seat();
    serial = wl_display_next_serial(device->pWlInputEvent->inputDevice().display());

    LOG_DEBUG("WaylandEvdevInputEvent",
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "button=" << button << ", "
              "state=" << state << ", "
              "time=" << time << ", "
              "button count=" << wlSeat->pointer->button_count << ", "
              "wayland seat=" << wlSeat << ", "
              "serial=" << serial);

    if (state == WL_POINTER_BUTTON_STATE_PRESSED)
    {
        if (wlSeat->pointer->button_count == 0)
        {
            wlSeat->pointer->grab_button = button;
            wlSeat->pointer->grab_time = time;
            wlSeat->pointer->grab_x = wlSeat->pointer->x;
            wlSeat->pointer->grab_y = wlSeat->pointer->y;
        }
        ++wlSeat->pointer->button_count;
    }
    else
    {
        if (wlSeat->pointer->button_count == 0)
        {
            LOG_WARNING("WaylandEvdevInputEvent",
                        "Unbalanced press/release event");

            return;
        }
        --wlSeat->pointer->button_count;
    }
    wlEvent.x = wl_fixed_to_int(wlSeat->pointer->x);
    wlEvent.y = wl_fixed_to_int(wlSeat->pointer->y);
    wlEvent.buttonState = state;
    wlEvent.serial = serial;
    wlEvent.button = button;

    device->pWlInputEvent->windowSystem().manageWLInputEvent(INPUT_DEVICE_POINTER,
        state == WL_POINTER_BUTTON_STATE_PRESSED ? INPUT_STATE_PRESSED :
        INPUT_STATE_RELEASED, &wlEvent, device->pWlInputEvent, device->seatName, device->displayID);

    if (wlSeat->pointer->button_count == 1)
    {
        wlSeat->pointer->grab_serial =
            wl_display_get_serial(device->pWlInputEvent->inputDevice().display());
    }
}

WaylandInputEvent *
WaylandEvdevInputEvent::getWaylandInputEvent(int displayID)
{
    WaylandInputEvent *pwlinputevent = NULL;

    struct evdev_input_device *device, *next;
    wl_list_for_each_safe(device, next, &m_deviceList, link)
    {
        if(device->displayID == displayID)
        {
            pwlinputevent = device->pWlInputEvent;
        }
    }
    return pwlinputevent;
}

void
WaylandEvdevInputEvent::notifyMotion(struct evdev_input_device *device,
                                     uint32_t time,
                                     wl_fixed_t fx, wl_fixed_t fy)
{
    WL_UNUSED(time);

    WLEvent         wlEvent;
    struct wl_seat *wlSeat = NULL;
    int             x;
    int             y;
    //int             old_x, old_y;
    int             w = 0;
    int             h = 0;

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "pending events flags=" << device->pending_events << ", "
                    "time=" << time);
        return;
    }

    inputEvent->getScreenResolution(device->displayID, w, h);

    wlSeat = device->pWlInputEvent->inputDevice().seat();


    x = wl_fixed_to_int(fx);
    y = wl_fixed_to_int(fy);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "wayland seat=" << wlSeat << ", "
              "pending events flags=" << device->pending_events << ", "
              "screen width=" << w << ", "
              "screen height=" << h << ", "
              "associated display=" << device->displayID << ", "
              "device"
              "x=" << x << ", "
              "y=" << y << ", "
              "time=" << time);

    if ( !(device->pending_events & EVDEV_RELATIVE_MOTION))
    {
        x = (x - device->abs.min_x) * w
            / (device->abs.max_x - device->abs.min_x)
            + 0; //device->output->x;
        y = (y - device->abs.min_y) * h
            / (device->abs.max_y - device->abs.min_y)
            + 0; //device->output->y;
    }

    if ((x < 0) || (x > w) || (y < 0) || (y > h))
    {
    	LOG_ERROR("WaylandEvdevInputEvent", "Touch point exceeds device max, "
              "device name=" << device->deviceName << ", "
              "associated display=" << device->displayID << ", "
              "Touch Point X=" << x << ", "
              "Touch point Y=" << y);
    }

    if (x < 0) x = 0;
    if (x > w) x = w;
    if (y < 0) y = 0;
    if (y > h) y = h;

    wlSeat->pointer->x = wl_fixed_from_int(x);
    wlSeat->pointer->y = wl_fixed_from_int(y);

    wlEvent.x = x;
    wlEvent.y = y;

    device->pWlInputEvent->windowSystem().manageWLInputEvent(
        INPUT_DEVICE_POINTER, INPUT_STATE_MOTION, &wlEvent,
        device->pWlInputEvent, device->seatName, device->displayID);
}

void
WaylandEvdevInputEvent::notifyAxis(struct evdev_input_device *device,
        uint32_t time, uint32_t axis, wl_fixed_t value)
{
    WL_UNUSED(time);
    WLEvent wlEvent;

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "axis=" << axis << ", "
                    "value=" << value << ", "
                    "time=" << time);
        return;
    }
    else
    {
        LOG_DEBUG("WaylandEvdevInputEvent",
                  "device name=" << device->deviceName << ", "
                  "device address=" << device << ", "
                  "axis=" << axis << ", "
                  "value=" << value << ", "
                  "time=" << time);
    }

    wlEvent.axis = axis;
    wlEvent.axisValue = value;

    device->pWlInputEvent->windowSystem().manageWLInputEvent(INPUT_DEVICE_POINTER,
        INPUT_STATE_AXIS, &wlEvent, device->pWlInputEvent, device->seatName, device->displayID);

}
void
WaylandEvdevInputEvent::notifyKey(struct evdev_input_device *device,
                                  uint32_t time, uint32_t key,
                                  enum wl_keyboard_key_state state,
                                  bool bUpdateAutomatic,
                                  bool treatPressAsRelease)
{
    WL_UNUSED(bUpdateAutomatic);

    WLEvent wlEvent;
    struct wl_seat *wlSeat = NULL;
    uint32_t *k;
    uint32_t *end;

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device << ", "
                    "key=" << key << ", "
                    "state=" << state << ", "
                    "time=" << time);
        return;
    }

    wlSeat = device->pWlInputEvent->inputDevice().seat();

    if (state == WL_KEYBOARD_KEY_STATE_PRESSED)
    {
        wlSeat->keyboard->grab_key = key;
        wlSeat->keyboard->grab_time = time;
    }

    // Find the key in wlSeat->keyboard->keys
    bool found = false;
    end = (uint32_t*)(((unsigned char*)wlSeat->keyboard->keys.data) + wlSeat->keyboard->keys.size);
    for (k = (uint32_t*)wlSeat->keyboard->keys.data; k < end; ++k)
    {
        // If the key is found
        if (*k == key)
        {
            found = true;
            if (state == WL_KEYBOARD_KEY_STATE_PRESSED && !treatPressAsRelease)
            {
                LOG_DEBUG("WaylandEvdevInputEvent",
                          "Key pressed and not treated as release, "
                          "device name=" << device->deviceName << ", "
                          "device address=" << device << ", "
                          "key=" << key << ", "
                          "state=" << state << ", "
                          "wlSeat=" << wlSeat << ", "
                          "time=" << time);
                return;
            }
            *k = *--end;
        }
    }
    wlSeat->keyboard->keys.size = (uint8_t*)end - (uint8_t*)wlSeat->keyboard->keys.data;

    // If the key was pressed and not already known
    if (state == WL_KEYBOARD_KEY_STATE_PRESSED && !found)
    {
        LOG_DEBUG("WaylandEvdevInputEvent",
                  "Key pressed but not known, "
                  "device name=" << device->deviceName << ", "
                  "device address=" << device << ", "
                  "key=" << key << ", "
                  "state=" << state << ", "
                  "wlSeat=" << wlSeat << ", "
                  "time=" << time);
        k = (uint32_t*)wl_array_add(&wlSeat->keyboard->keys, sizeof(*k));
        *k = key;
    }

    wlEvent.keyCode = key;
    wlEvent.keyState = state;

    LOG_DEBUG("WaylandEvdevInputEvent",
              "Managing event, "
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "key=" << key << ", "
              "state=" << state << ", "
              "wlSeat=" << wlSeat << ", "
              "time=" << time);

    device->pWlInputEvent->windowSystem().manageWLInputEvent(INPUT_DEVICE_KEYBOARD,
        state == WL_KEYBOARD_KEY_STATE_PRESSED ? INPUT_STATE_PRESSED
                                               : INPUT_STATE_RELEASED, &wlEvent,
                                device->pWlInputEvent, device->seatName, device->displayID);
}

void
WaylandEvdevInputEvent::notifyTouch(struct evdev_input_device *device)
{
    WLEvent         wlEvent;
    InputEventState eventState = INPUT_STATE_OTHER;
	int scrnWidth = 0;
	int scrnHeight = 0;
	int devCoordX;
	int devCoordY;

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("WaylandEvdevInputEvent",
                    "Wayland evdev input event NULL for, "
                    "device name=" << device->deviceName << ", "
                    "device address=" << device);
        return;
    }

    inputEvent->getScreenResolution(device->displayID, scrnWidth, scrnHeight);

    LOG_DEBUG("WaylandEvdevInputEvent",
              "device name=" << device->deviceName << ", "
              "device address=" << device << ", "
              "associated display=" << device->displayID << ", "
              "screen width=" << scrnWidth << ", "
              "screen height=" << scrnHeight << ", ");

    devCoordX = device->mt.x[device->mt.slot];
    devCoordY = device->mt.y[device->mt.slot];

    if (devCoordX > device->abs.max_x)
    {
        LOG_ERROR("WaylandEvdevInputEvent", "Touch point X exceeds device max, "
                  "device name=" << device->deviceName << ", "
                  "associated display=" << device->displayID << ", "
                  "Touch Point X=" << devCoordX << ", "
                  "dev Max Touch point X=" << device->abs.max_x);
    	devCoordX = device->abs.max_x;
    }
    else if (devCoordX < device->abs.min_x)
    {
        LOG_ERROR("WaylandEvdevInputEvent", "Touch point X less than device min, "
                  "device name=" << device->deviceName << ", "
                  "associated display=" << device->displayID << ", "
                  "Touch Point X=" << devCoordX << ", "
                  "dev Min Touch point X=" << device->abs.min_x);
    	devCoordX = device->abs.min_x;
    }

    if (devCoordY > device->abs.max_y)
    {
        LOG_ERROR("WaylandEvdevInputEvent", "Touch point Y exceeds device max, "
                  "device name=" << device->deviceName << ", "
                  "associated display=" << device->displayID << ", "
                  "Touch Point Y=" << devCoordY << ", "
                  "dev Max Touch point Y=" << device->abs.max_y);
    	devCoordY = device->abs.max_y;
    }
    else if (devCoordY < device->abs.min_y)
    {
        LOG_ERROR("WaylandEvdevInputEvent", "Touch point Y less than device min, "
                  "device name=" << device->deviceName << ", "
                  "associated display=" << device->displayID << ", "
                  "Touch Point Y=" << devCoordY << ", "
                  "dev Min Touch point Y=" << device->abs.min_y);
    	devCoordY = device->abs.min_y;
    }

    wlEvent.x = (devCoordX - device->abs.min_x) *
                 scrnWidth /
                (device->abs.max_x - device->abs.min_x) +
                0; //device->output->x;
    wlEvent.y = (devCoordY - device->abs.min_y) *
                 scrnHeight /
                (device->abs.max_y - device->abs.min_y) +
                0; //device->output->y;
    wlEvent.touchId = device->mt.slot;
    if (device->pending_events & EVDEV_ABSOLUTE_MT_DOWN)
    {
        wlEvent.touchType = WL_TOUCH_DOWN;
        eventState = INPUT_STATE_PRESSED;
    }
    else if (device->pending_events & EVDEV_ABSOLUTE_MT_MOTION)
    {
        wlEvent.touchType = WL_TOUCH_MOTION;
        eventState = INPUT_STATE_MOTION;
    }
    else if (device->pending_events & EVDEV_ABSOLUTE_MT_UP)
    {
        wlEvent.x = 0;
        wlEvent.y = 0;
        wlEvent.touchType = WL_TOUCH_UP;
        eventState = INPUT_STATE_RELEASED;
    }
    else
    {
        return;
    }

    device->pWlInputEvent->windowSystem().manageWLInputEvent(INPUT_DEVICE_TOUCH,
        eventState, &wlEvent, device->pWlInputEvent, device->seatName, device->displayID);
   
}

int WaylandEvdevInputEvent::getScreenDefaultWidth()
{
    return m_screenWidth;
}

int WaylandEvdevInputEvent::getScreenDefaultHeight()
{
    return m_screenHeight;
}

struct wl_event_loop* WaylandEvdevInputEvent::getEventLoop()
{
    return m_eventLoop;
}
